/*

 BinaryStream Class for C++
 (c) 2004 Jonathan Bettencourt / Kreative Korporation

 This imitates the BinaryStream class from RealBasic.
 You create a BinaryStream class from a certain file, and
 you can do random access and read and write stuff.

 The methods for reading and writing are similar to the
 methods for getting and setting things in a memoryblock.

 Sorry the documentation on this one isn't as nicely done
 as on MemoryBlock, but whatever.


 This code is under the MIT license.

 Permission is hereby granted, free of charge, to any person obtaining a copy of
 this software and associated documentation files (the "Software"), to deal in the
 Software without restriction, including without limitation the rights to use,
 copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
 Software, and to permit persons to whom the Software is furnished to do so,
 subject to the following conditions:

 The above copyright notice and this permission notice shall be included in all copies
 or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/


#include <string>
#include <fstream>
#include <iostream>
#include "binarystream.h"
using namespace std;

binarystream::binarystream(void)
{
	setDefaultEndianness();
	isopen = false;
	mypos = 0;
}

binarystream::binarystream(char * filename, bool writeAccess, bool clearFile)
{
	setDefaultEndianness();
	isopen = false;
	mypos = 0;
	open(filename, writeAccess, clearFile);
}

binarystream::binarystream(const string & filename, bool writeAccess, bool clearFile)
{
	setDefaultEndianness();
	isopen = false;
	mypos = 0;
	open(filename.c_str(), writeAccess, clearFile);
}

/* Copy Constructor Disabled Because the Compiler Decided to Hate Me...
binarystream::binarystream(const binarystream& b)
{
	mystream = b.mystream;  you can't do this...
	b.open(savefname, savewritacc, false);
	littleEndian = b.littleEndian;
}
*/

binarystream::~binarystream(void)
{
	close();
}

/* Assignment Disabled Because the Compiler Decided to Hate Me...
const binarystream& binarystream::operator = (const binarystream& b)
{
	close();
	mystream = b.mystream;
	littleEndian = b.littleEndian;
	return *this;
}
*/

void binarystream::open(const char * filename, bool writeAccess, bool clearFile)
{
	if (isopen) { throw((binarystream_exception) alreadyOpen); }
	savefname = new char[strlen(filename)+1];
	memcpy(savefname, filename, strlen(filename)+1);
	savewritacc = writeAccess;
	if (writeAccess)
	{
		if (clearFile)
		{
			mystream.open(filename, ios::in | ios::out | ios::trunc | ios::binary);
		} else {
			mystream.open(filename, ios::in | ios::out | ios::binary);
		}
	} else {
		mystream.open(filename, ios::in | ios::binary);
	}
	if (!mystream.is_open())
	{
		isopen = false;
		throw((binarystream_exception) openFailed);
	} else {
		isopen = true;
		mypos = 0;
	}
}

void binarystream::open(const string & filename, bool writeAccess, bool clearFile)
{
	open(filename.c_str(), writeAccess, clearFile);
}

void binarystream::close(void)
{
	if (isopen)
	{
		mystream.close();
		isopen = false;
		mypos = 0;
	}
}

void binarystream::setDefaultEndianness(void)
{
	short int a = 0x0001;
	char * b = (char *) &a;
	littleEndian = (b[0]?true:false);
}

char * binarystream::read(int l)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	char * buffer;
	mystream.clear(); /* clear up any errors that have occurred */
	buffer = new char[l];
	mystream.read(buffer, l);
	mystream.seekp(mystream.tellg(), ios::beg);
	return buffer;
}

void binarystream::write(char * buffer, int l)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	mystream.clear();
	mystream.write(buffer, l);
	mystream.seekg(mystream.tellp(), ios::beg);
}

void binarystream::write(string buffer, int l)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	mystream.clear();
	mystream.write(buffer.c_str(), l);
	mystream.seekg(mystream.tellp(), ios::beg);
}

bool binarystream::eof(void)
{
	return (mystream.eof() | (getLength() == getPosition()));
}

bool binarystream::ok(void)
{
	return (isopen & !(mystream.bad() | mystream.fail()));
}

void binarystream::reset(void)
{
	mystream.clear();
}

unsigned long int binarystream::getLength(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	unsigned long int l;
	unsigned long int oldPos = mystream.tellg();
	mystream.seekg((streampos)0, ios::end);
	l = mystream.tellg();
	mystream.seekg((streampos)oldPos, ios::beg);
	return l;
}

void binarystream::setLength(unsigned long int l)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	unsigned long int pl;
	unsigned long int i;
	char * nl;
	char * tmp;
	unsigned long int oldPos = mystream.tellg();
	unsigned long int oldPos2 = mystream.tellp();
	if (savewritacc)
	{
		mystream.clear();
		mystream.seekg((streampos)0, ios::end);
		pl = mystream.tellg();
		if (pl < l)
		{
			nl = new char[1];
			nl[0] = 0;
			mystream.seekp((streampos)0, ios::end);
			for (i=pl; i<l; ++i)
			{
				mystream.write(nl, 1);
			}
			mystream.seekp((streampos)oldPos2, ios::beg);
			delete [] nl;
		}
		if (pl > l)
		{
			mystream.seekg((streampos)0, ios::beg);
			tmp = new char[l];
			mystream.read(tmp, l);
			mystream.close();
			isopen = false; // <--- Otherwise, we'll throw an exception.
			open(savefname, true, true);
			if (ok())
			{
				mystream.write(tmp, l);
				mystream.seekp((streampos)oldPos2, ios::beg);
			}
			delete [] tmp;
		}
		mystream.seekg((streampos)oldPos, ios::beg);
	}
}

unsigned long int binarystream::getPosition(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	unsigned long int pos1 = mystream.tellg();
	unsigned long int pos2 = mystream.tellp();
	return ((pos1 + pos2) / 2);
}

void binarystream::setPosition(unsigned long int pos)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	mystream.seekg((streampos)pos, ios::beg);
	mystream.seekp((streampos)pos, ios::beg);
}

fstream& binarystream::getFStream(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	return mystream;
}

/*
 REALBASIC READ/WRITE FUNCTIONS
 These imitate the methods of BinaryStream objects in RealBasic,
 using the same terminology as MemoryBlock objects in RealBasic.
 They use a general data type (bool, int, double, char*, or string)
 for all data types, regardless of size. These are mostly compiler-
 and platform-independent.
*/

bool binarystream::readBooleanValue(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	char * t;
	bool r;
	t = new char[1];
	mystream.read(t, 1);
	r = (t != 0);
	delete [] t;
	mystream.seekp(mystream.tellg(), ios::beg);
	return r;
}

int binarystream::readByte(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	char * t;
	int r;
	t = new char[1];
	mystream.read(t, 1);
	r = t[0];
	delete [] t;
	mystream.seekp(mystream.tellg(), ios::beg);
	return r;
}

char * binarystream::readCString(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	const int gran = 50;
	char c, *b, *t;
	int loc = 0;
	int cur_size = gran;
	b = new char[gran];
	c = readChar();
	do
	{
		b[loc++] = c;
		if (loc == (cur_size-1))
		{
			cur_size += gran;
			t = new char[cur_size];
			strcpy(t,b);
			delete [] b;
			b = t;
		}
	} while (c = readChar());
	t = new char[strlen(b) + 1];
	strcpy(t,b);
	delete [] b;
	mystream.seekp(mystream.tellg(), ios::beg);
	return t;
}

string binarystream::readCStringAsString(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	string s;
	const int gran = 50;
	char c, *b, *t;
	int loc = 0;
	int cur_size = gran;
	b = new char[gran];
	c = readChar();
	do
	{
		b[loc++] = c;
		if (loc == (cur_size-1))
		{
			cur_size += gran;
			t = new char[cur_size];
			strcpy(t,b);
			delete [] b;
			b = t;
		}
	} while (c = readChar());
	t = new char[strlen(b) + 1];
	strcpy(t,b);
	delete [] b;
	s = t;
	delete [] t;
	mystream.seekp(mystream.tellg(), ios::beg);
	return s;
}

double binarystream::readDoubleValue(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	double t;
	bconverter *c = new bconverter;
	mystream.read(c->charv, sizeof(double));
	mystream.seekp(mystream.tellg(), ios::beg);
	t = c->doublev;
	delete c;
	return t;
}

int binarystream::readLong(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	int i;
	char * t;
	t = new char[4];
	mystream.read(t, 4);
	if (littleEndian)
	{
		i = t[0] + (256 * t[1]) + (65536 * t[2]) + (16777216 * t[3]);
	} else {
		i = t[3] + (256 * t[2]) + (65536 * t[1]) + (16777216 * t[0]);
	}
	delete [] t;
	mystream.seekp(mystream.tellg(), ios::beg);
	return i;
}

char * binarystream::readPString(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	int sl = readChar();
	char * t = new char[sl+1];
	t[0] = sl;
	mystream.read(t+1, sl);
	mystream.seekp(mystream.tellg(), ios::beg);
	return t;
}

char * binarystream::readPStringAsCString(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	int sl = readChar();
	char * t = new char[sl+1];
	int i;
	t[sl] = 0;
	if (sl>0)
	{
		mystream.read(t, sl);
		for (i=0; i<sl; ++i)
		{
			if (t[i] == 0)
			{
				t[i] = 32;
			}
		}
	}
	mystream.seekp(mystream.tellg(), ios::beg);
	return t;
}

string binarystream::readPStringAsString(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	int sl = readChar();
	char * t = new char[sl+1];
	int i;
	string s;
	t[sl] = 0;
	if (sl>0)
	{
		mystream.read(t, sl);
		for (i=0; i<sl; ++i)
		{
			if (t[i] == 0)
			{
				t[i] = 32;
			}
		}
	}
	s = t;
	delete [] t;
	mystream.seekp(mystream.tellg(), ios::beg);
	return s;
}

int binarystream::readShort(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	int i;
	char * t;
	t = new char[2];
	mystream.read(t, 2);
	if (littleEndian)
	{
		i = t[0] + (256 * t[1]);
	} else {
		i = t[1] + (256 * t[0]);
	}
	delete [] t;
	mystream.seekp(mystream.tellg(), ios::beg);
	return (i<32768)?i:(i-65536);
}

double binarystream::readSingleValue(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	float t;
	bconverter *c = new bconverter;
	mystream.read(c->charv, sizeof(float));
	mystream.seekp(mystream.tellg(), ios::beg);
	t = c->floatv;
	delete c;
	return (double)t;
}

char * binarystream::readStringValue(int sl)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	char * t;
	t = new char[sl];
	mystream.read(t, sl);
	mystream.seekp(mystream.tellg(), ios::beg);
	return t;
}

char * binarystream::readStringValueAsCString(int sl)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	char * t;
	int i;
	t = new char[sl+1];
	t[sl] = 0;
	if (sl>0)
	{
		mystream.read(t, sl);
		for (i=0; i<sl; ++i)
		{
			if (t[i] == 0)
			{
				t[i] = 32;
			}
		}
	}
	mystream.seekp(mystream.tellg(), ios::beg);
	return t;
}

string binarystream::readStringValueAsString(int sl)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	char * t;
	string s;
	int i;
	t = new char[sl+1];
	t[sl] = 0;
	if (sl>0)
	{
		mystream.read(t, sl);
		for (i=0; i<sl; ++i)
		{
			if (t[i] == 0)
			{
				t[i] = 32;
			}
		}
	}
	s = t;
	delete [] t;
	mystream.seekp(mystream.tellg(), ios::beg);
	return s;
}

int binarystream::readUShort(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	int i;
	char * t;
	t = new char[2];
	mystream.read(t, 2);
	if (littleEndian)
	{
		i = t[0] + (256 * t[1]);
	} else {
		i = t[1] + (256 * t[0]);
	}
	delete [] t;
	mystream.seekp(mystream.tellg(), ios::beg);
	return i;
}

void binarystream::writeBooleanValue(bool value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	char * t;
	t = new char[1];
	t[0] = value?1:0;
	mystream.write(t, 1);
	delete [] t;
	mystream.seekg(mystream.tellp(), ios::beg);
}

void binarystream::writeByte(int value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	char * t;
	t = new char[1];
	t[0] = value;
	mystream.write(t, 1);
	delete [] t;
	mystream.seekg(mystream.tellp(), ios::beg);
}

void binarystream::writeCString(const char * value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	mystream.write(value, strlen(value)+1);
	mystream.seekg(mystream.tellp(), ios::beg);
}

void binarystream::writeCStringAsString(string value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	mystream.write(value.c_str(), value.size()+1);
	mystream.seekg(mystream.tellp(), ios::beg);
}

void binarystream::writeDoubleValue(double value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	bconverter *c = new bconverter;
	c->doublev = value;
	mystream.write(c->charv, sizeof(value));
	mystream.seekg(mystream.tellp(), ios::beg);
	delete c;
}

void binarystream::writeLong(int value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	char * t;
	int a = (value / 16777216) % 256;
	int b = (value / 65536) % 256;
	int c = (value / 256) % 256;
	int d = value % 256;
	t = new char[4];
	if (littleEndian)
	{
		t[0] = d;
		t[1] = c;
		t[2] = b;
		t[3] = a;
	} else {
		t[0] = a;
		t[1] = b;
		t[2] = c;
		t[3] = d;
	}
	mystream.write(t, 4);
	delete [] t;
	mystream.seekg(mystream.tellp(), ios::beg);
}

void binarystream::writePString(char * value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	int sl = value[0];
	mystream.write(value, sl+1);
	mystream.seekg(mystream.tellp(), ios::beg);
}

void binarystream::writePStringAsCString(char * value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	writeByte(strlen(value));
	mystream.write(value, strlen(value));
	mystream.seekg(mystream.tellp(), ios::beg);
}

void binarystream::writePStringAsString(string value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	writeByte(value.size());
	mystream.write(value.c_str(), value.size());
	mystream.seekg(mystream.tellp(), ios::beg);
}

void binarystream::writeShort(int value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	char * tmp;
	int t = value;
	int a;
	int b;
	if (t>65536) { t-=65536; }
	if (t<65536) { t+=65536; }
	a = (t / 256) % 256;
	b = t % 256;
	tmp = new char[2];
	if (littleEndian)
	{
		tmp[0] = b;
		tmp[1] = a;
	} else {
		tmp[0] = a;
		tmp[1] = b;
	}
	mystream.write(tmp, 2);
	delete [] tmp;
	mystream.seekg(mystream.tellp(), ios::beg);
}

void binarystream::writeSingleValue(double value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	bconverter *c = new bconverter;
	c->floatv = (float)value;
	mystream.write(c->charv, sizeof(float));
	mystream.seekg(mystream.tellp(), ios::beg);
	delete c;
}

void binarystream::writeStringValue(int sl, char * value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	mystream.write(value, sl);
	mystream.seekg(mystream.tellp(), ios::beg);
}

void binarystream::writeStringValueAsCString(int sl, char * value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	int sl2;
	int i;
	sl2 = min(strlen(value)+1, sl);
	mystream.write(value, sl2);
	if (sl2 < sl)
	{
		for (i=sl2; i<sl; ++i)
		{
			writeByte(0);
		}
	}
	mystream.seekg(mystream.tellp(), ios::beg);
}

void binarystream::writeStringValueAsString(int sl, string value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	char * t;
	t = new char[value.size() + 1];
	strcpy(t, value.c_str());
	writeStringValueAsCString(sl, t);
	delete [] t;
	mystream.seekg(mystream.tellp(), ios::beg);
}

void binarystream::writeUShort(int value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	char * tmp;
	int t = value;
	int a;
	int b;
	if (t>65536) { t-=65536; }
	if (t<65536) { t+=65536; }
	a = (t / 256) % 256;
	b = t % 256;
	tmp = new char[2];
	if (littleEndian)
	{
		tmp[0] = b;
		tmp[1] = a;
	} else {
		tmp[0] = a;
		tmp[1] = b;
	}
	mystream.write(tmp, 2);
	delete [] tmp;
	mystream.seekg(mystream.tellp(), ios::beg);
}

/*
 C++ READ/WRITE FUNCTIONS
 These use specific C++ data types. They are compiler- and platform-dependent,
 like regular C++ variables.
*/

bool binarystream::readBool(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	bool t;
	bconverter *c = new bconverter;
	mystream.read(c->charv, sizeof(bool));
	mystream.seekp(mystream.tellg(), ios::beg);
	t = c->boolv;
	delete c;
	return t;
}

char binarystream::readChar(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	char t;
	bconverter *c = new bconverter;
	mystream.read(c->charv, sizeof(char));
	mystream.seekp(mystream.tellg(), ios::beg);
	t = c->charv2;
	delete c;
	return t;
}

signed char binarystream::readSignedChar(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	signed char t;
	bconverter *c = new bconverter;
	mystream.read(c->charv, sizeof(signed char));
	mystream.seekp(mystream.tellg(), ios::beg);
	t = c->scharv;
	delete c;
	return t;
}

int binarystream::readInt(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	int t;
	bconverter *c = new bconverter;
	mystream.read(c->charv, sizeof(int));
	mystream.seekp(mystream.tellg(), ios::beg);
	t = c->intv;
	delete c;
	return t;
}

short int binarystream::readShortInt(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	short int t;
	bconverter *c = new bconverter;
	mystream.read(c->charv, sizeof(short int));
	mystream.seekp(mystream.tellg(), ios::beg);
	t = c->sintv;
	delete c;
	return t;
}

long int binarystream::readLongInt(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	long int t;
	bconverter *c = new bconverter;
	mystream.read(c->charv, sizeof(long int));
	mystream.seekp(mystream.tellg(), ios::beg);
	t = c->lintv;
	delete c;
	return t;
}

unsigned short int binarystream::readUnsignedShortInt(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	unsigned short int t;
	bconverter *c = new bconverter;
	mystream.read(c->charv, sizeof(unsigned short int));
	mystream.seekp(mystream.tellg(), ios::beg);
	t = c->usintv;
	delete c;
	return t;
}

unsigned long int binarystream::readUnsignedLongInt(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	unsigned long int t;
	bconverter *c = new bconverter;
	mystream.read(c->charv, sizeof(unsigned long int));
	mystream.seekp(mystream.tellg(), ios::beg);
	t = c->ulintv;
	delete c;
	return t;
}

float binarystream::readFloat(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	float t;
	bconverter *c = new bconverter;
	mystream.read(c->charv, sizeof(float));
	mystream.seekp(mystream.tellg(), ios::beg);
	t = c->floatv;
	delete c;
	return t;
}

double binarystream::readDouble(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	double t;
	bconverter *c = new bconverter;
	mystream.read(c->charv, sizeof(double));
	mystream.seekp(mystream.tellg(), ios::beg);
	t = c->doublev;
	delete c;
	return t;
}

long double binarystream::readLongDouble(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	long double t;
	bconverter *c = new bconverter;
	mystream.read(c->charv, sizeof(long double));
	mystream.seekp(mystream.tellg(), ios::beg);
	t = c->ldoublev;
	delete c;
	return t;
}

wchar_t binarystream::readWchar_t(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	wchar_t t;
	bconverter *c = new bconverter;
	mystream.read(c->charv, sizeof(wchar_t));
	mystream.seekp(mystream.tellg(), ios::beg);
	t = c->wchartv;
	delete c;
	return t;
}

string binarystream::readString(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	char * t;
	char * tt;
	string s;
	int op = mystream.tellg();
	int cp = mystream.tellg();
	tt = read(1);
	while (tt[0] != 0)
	{
		cp++;
		tt = read(1);
	}
	/* cp now points to our zero terminator */
	mystream.seekg(op, ios::beg);
	t = new char[(cp-op)+1];
	mystream.read(t, (cp-op)+1);
	delete [] tt;
	s = t;
	delete [] t;
	mystream.seekp(mystream.tellg(), ios::beg);
	return s;
}

char * binarystream::readCharStar(void)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	char * t;
	char * tt;
	int op = mystream.tellg();
	int cp = mystream.tellg();
	tt = read(1);
	while (tt[0] != 0)
	{
		cp++;
		tt = read(1);
	}
	/* cp now points to our zero terminator */
	mystream.seekg(op, ios::beg);
	t = new char[(cp-op)+1];
	mystream.read(t, (cp-op)+1);
	delete [] tt;
	mystream.seekp(mystream.tellg(), ios::beg);
	return t;
}

void binarystream::writeBool(bool value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	bconverter *c = new bconverter;
	c->boolv = value;
	mystream.write(c->charv, sizeof(value));
	mystream.seekg(mystream.tellp(), ios::beg);
	delete c;
}

void binarystream::writeChar(char value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	bconverter *c = new bconverter;
	c->charv2 = value;
	mystream.write(c->charv, sizeof(value));
	mystream.seekg(mystream.tellp(), ios::beg);
	delete c;
}

void binarystream::writeSignedChar(signed char value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	bconverter *c = new bconverter;
	c->scharv = value;
	mystream.write(c->charv, sizeof(value));
	mystream.seekg(mystream.tellp(), ios::beg);
	delete c;
}

void binarystream::writeInt(int value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	bconverter *c = new bconverter;
	c->intv = value;
	mystream.write(c->charv, sizeof(value));
	mystream.seekg(mystream.tellp(), ios::beg);
	delete c;
}

void binarystream::writeShortInt(short int value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	bconverter *c = new bconverter;
	c->sintv = value;
	mystream.write(c->charv, sizeof(value));
	mystream.seekg(mystream.tellp(), ios::beg);
	delete c;
}

void binarystream::writeLongInt(long int value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	bconverter *c = new bconverter;
	c->lintv = value;
	mystream.write(c->charv, sizeof(value));
	mystream.seekg(mystream.tellp(), ios::beg);
	delete c;
}

void binarystream::writeUnsignedShortInt(unsigned short int value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	bconverter *c = new bconverter;
	c->usintv = value;
	mystream.write(c->charv, sizeof(value));
	mystream.seekg(mystream.tellp(), ios::beg);
	delete c;
}

void binarystream::writeUnsignedLongInt(unsigned long int value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	bconverter *c = new bconverter;
	c->ulintv = value;
	mystream.write(c->charv, sizeof(value));
	mystream.seekg(mystream.tellp(), ios::beg);
	delete c;
}

void binarystream::writeFloat(float value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	bconverter *c = new bconverter;
	c->floatv = value;
	mystream.write(c->charv, sizeof(value));
	mystream.seekg(mystream.tellp(), ios::beg);
	delete c;
}

void binarystream::writeDouble(double value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	bconverter *c = new bconverter;
	c->doublev = value;
	mystream.write(c->charv, sizeof(value));
	mystream.seekg(mystream.tellp(), ios::beg);
	delete c;
}

void binarystream::writeLongDouble(long double value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	bconverter *c = new bconverter;
	c->ldoublev = value;
	mystream.write(c->charv, sizeof(value));
	mystream.seekg(mystream.tellp(), ios::beg);
	delete c;
}

void binarystream::writeWchar_t(wchar_t value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	bconverter *c = new bconverter;
	c->wchartv = value;
	mystream.write(c->charv, sizeof(value));
	mystream.seekg(mystream.tellp(), ios::beg);
	delete c;
}

void binarystream::writeString(string value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	mystream.write(value.c_str(), value.size()+1);
	mystream.seekg(mystream.tellp(), ios::beg);
}

void binarystream::writeCharStar(char * value)
{
	if (!isopen) { throw((binarystream_exception) notOpen); } else if (mystream.bad() | mystream.fail() | mystream.eof()) { mystream.clear(); }
	mystream.write(value, strlen(value)+1);
	mystream.seekg(mystream.tellp(), ios::beg);
}